﻿using Microsoft.AspNetCore;
using Microsoft.AspNetCore.Hosting;
using Microsoft.Extensions.Configuration;
using ORC.TF.Logging;
using Serilog;
using Serilog.Events;
using System;
using System.Diagnostics;
using System.IO;

namespace ORC.TF.Custom.AdventureWorksSample
{
    public class Program
    {
        public static IConfiguration Configuration { get; private set; }
        public static bool IsDevelopmentBuild { get; private set; }

        public static int Main(string[] args)
        {
            string environment = Environment.GetEnvironmentVariable("ASPNETCORE_ENVIRONMENT") ?? "Development";
            IsDevelopmentBuild = environment == "Development";

            Configuration = new ConfigurationBuilder()
                .SetBasePath(Directory.GetCurrentDirectory())
                .AddJsonFile("appsettings.json", optional: false, reloadOnChange: true)
                .AddJsonFile($"appsettings.{environment}.json", optional: true)
                .AddEnvironmentVariables()
                .Build();

            string connectionString = Configuration.GetConnectionString("ThoughtFarmer");

            // Set the default logger here so we can capture and log startup errors. We have no config or dependencies registered yet, so it is hardcoded to try and use all sinks.
            // IMPORTANT: We must change this logger after startup is complete (Startup.cs), otherwise we this logger config is out of sync with the site's logging settings.
            // We also dial down the minimum level for some sources, because by setting this static logger we are affecting up ALL sources, since Serilog integrates tightly
            // with .NET logging infrastructure and as such captures logs from all sources (ie. Hangfire, Microsoft framework internals, etc.)
            Log.Logger = new LoggerConfiguration()
                .MinimumLevel.Debug()
                .MinimumLevel.Override("Microsoft", LogEventLevel.Warning)
                .MinimumLevel.Override("System", LogEventLevel.Warning)
                .MinimumLevel.Override("Hangfire", LogEventLevel.Warning)
                .WriteTo.Seq("http://localhost:5341") // this sink may not be available, it doesn't matter we try and use it
                .WriteTo.MSSqlServer(connectionString, "SystemLogs", columnOptions: SqlServerSettings.ColumnOptions)
                .CreateLogger();

            try
            {
                Log.Information("Web host starting");
                CreateWebHostBuilder(args).Build().Run();
                return 0;
            }
            catch (Exception ex)
            {
                Log.Fatal(ex, "Host terminated unexpectedly");
                return 1;
            }
            finally
            {
                Log.Information("Web host shutting down");
                Log.CloseAndFlush();
            }
        }

        public static IWebHostBuilder CreateWebHostBuilder(string[] args)
        {
            return WebHost.CreateDefaultBuilder(args)
                .UseIISIntegration()
                .UseStartup<Startup>()
                .UseConfiguration(Configuration)
                .UseSerilog();
        }
    }
}
